<?php
session_start();
header('Content-Type: application/json');

// Mostrar errores para depuración
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar si el usuario está logueado
if (!isset($_SESSION['usuario_logueado']) || $_SESSION['usuario_logueado'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Usuario no autenticado']);
    exit();
}

// Incluir conexión a base de datos
require_once 'conexion.php';

try {
    // Obtener datos JSON del request
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input) {
        throw new Exception('Datos de entrada inválidos');
    }

    $reserva_id = $input['reserva_id'] ?? null;
    $usuario_id = $_SESSION['usuario_id'];

    if (!$reserva_id) {
        throw new Exception('ID de reserva requerido');
    }

    // Verificar que la reserva existe y pertenece al usuario
    $stmt = $conexion->prepare("
        SELECT r.*, p.numero as plaza_numero, p.tipo as plaza_tipo
        FROM reservas r
        JOIN plazas p ON r.plaza_id = p.id
        WHERE r.id = ? AND r.usuario_id = ? AND r.estado = 'activa'
    ");
    $stmt->bind_param("ii", $reserva_id, $usuario_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $reserva = $result->fetch_assoc();

    if (!$reserva) {
        throw new Exception('Reserva no encontrada o no autorizada');
    }

    // Verificar que la reserva no haya comenzado (mínimo 1 hora antes)
    $fecha_hora_inicio = $reserva['fecha_reserva'] . ' ' . $reserva['hora_inicio'];
    $limite_cancelacion = date('Y-m-d H:i:s', strtotime($fecha_hora_inicio . ' -1 hour'));

    if (date('Y-m-d H:i:s') > $limite_cancelacion) {
        throw new Exception('No se puede cancelar la reserva. Debe cancelarse al menos 1 hora antes del inicio.');
    }

    // Iniciar transacción
    $conexion->begin_transaction();

    try {
        // Cancelar la reserva
        $stmt = $conexion->prepare("
            UPDATE reservas 
            SET estado = 'cancelada' 
            WHERE id = ? AND usuario_id = ?
        ");
        $stmt->bind_param("ii", $reserva_id, $usuario_id);
        $stmt->execute();

        if ($stmt->affected_rows === 0) {
            throw new Exception('No se pudo cancelar la reserva');
        }

        $conexion->commit();

        echo json_encode([
            'success' => true,
            'message' => 'Reserva cancelada exitosamente',
            'reserva_cancelada' => [
                'id' => $reserva['id'],
                'plaza_numero' => $reserva['plaza_numero'],
                'fecha_reserva' => $reserva['fecha_reserva'],
                'hora_inicio' => $reserva['hora_inicio']
            ]
        ]);
    } catch (Exception $e) {
        $conexion->rollback();
        throw $e;
    }

} catch (Exception $e) {
    error_log("Error en cancelar_reserva.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
